/*
  toolhelp implementation for Windows NT/Windows9X
  written by alexander yaworsky
  june '99
*/

#include <windows.h>

#include "switches.h"
#include "stdlib.h"
#include "paths.h"
#include "toolhelp.h"


typedef HANDLE WINAPI (*TCreateToolhelp32Snapshot)( DWORD dwFlags, DWORD th32ProcessID );
typedef BOOL WINAPI (*TProcess32First)( HANDLE hSnapshot, LPPROCESSENTRY32 lppe );
typedef BOOL WINAPI (*TProcess32Next)( HANDLE hSnapshot, LPPROCESSENTRY32 lppe );
typedef BOOL WINAPI (*TThread32First)( HANDLE hSnapshot, LPTHREADENTRY32 lpte );
typedef BOOL WINAPI (*TThread32Next)( HANDLE hSnapshot, LPTHREADENTRY32 lpte );
typedef BOOL WINAPI (*TCloseHandle)( HANDLE hObject );

typedef DWORD WINAPI (*TNtQuerySystemInformation)( DWORD ObjectType,
                         LPBYTE Buffer, DWORD BufferSize, DWORD Unknown );
typedef LONG WINAPI (*TRtlUnicodeStringToAnsiString)(
                         void* DestinationString,
                         void* SourceString,
                         BOOL AllocateDestinationString );
typedef VOID WINAPI (*TRtlInitUnicodeString)(
                         void* DestinationString,
                         void* SourceString );
typedef VOID WINAPI (*TRtlFreeAnsiString)( void* AnsiString );


BOOL WindowsNT = FALSE;

static TCreateToolhelp32Snapshot    pCreateToolhelp32Snapshot = NULL;
static TProcess32First              pProcess32First = NULL;
static TProcess32Next               pProcess32Next = NULL;
static TThread32First               pThread32First = NULL;
static TThread32Next                pThread32Next = NULL;
static TCloseHandle                 pCloseHandle = NULL;

static TNtQuerySystemInformation       pNtQuerySystemInformation = NULL;
static TRtlUnicodeStringToAnsiString   pRtlUnicodeStringToAnsiString = NULL;
static TRtlInitUnicodeString           pRtlInitUnicodeString = NULL;
static TRtlFreeAnsiString              pRtlFreeAnsiString = NULL;


#define SNAP_EPROC_CURRENT_PROC     0
#define SNAP_ETHR_CURRENT_PROC      4
#define SNAP_ETHR_CURRENT_THREAD    8
#define SNAP_RESERVED_SZ           12


static HANDLE WINAPI NtThCreateToolhelp32Snapshot(
                           DWORD dwFlags, DWORD th32ProcessID )
  {
    DWORD     Rc;
    int       BufSz;
    char      *SnapBuf;

    dwFlags = 0; th32ProcessID = 0;
    SnapBuf = NULL; BufSz = 0;
    do {
      BufSz += 4096;
      if( SnapBuf != NULL ) LocalFree( SnapBuf );
      SnapBuf = (char*) LocalAlloc( LPTR, BufSz );
      if( SnapBuf == NULL ) {
        SetLastError( ERROR_OUTOFMEMORY );
        return INVALID_HANDLE_VALUE;
      }
      Rc = pNtQuerySystemInformation( 5, SnapBuf + SNAP_RESERVED_SZ,
                                      BufSz - SNAP_RESERVED_SZ, 0 );
    } while( Rc == 0xC0000004 );
    if( Rc != 0 ) {
      if( SnapBuf != NULL ) LocalFree( SnapBuf );
      SetLastError( Rc );
      return INVALID_HANDLE_VALUE;
    }
    /* first DWORD is used by process enumerator; this is offset of
       current entry in snapshot, value -1 means no more processes
       second and third DWORDs are used by thread enumerator, second DWORD
       also is offset of current entry in snapshot, third DWORD is current
       thread number within process
    */
    *(DWORD*)(SnapBuf + SNAP_EPROC_CURRENT_PROC) = SNAP_RESERVED_SZ;
    *(DWORD*)(SnapBuf + SNAP_ETHR_CURRENT_PROC) = SNAP_RESERVED_SZ;
    *(DWORD*)(SnapBuf + SNAP_ETHR_CURRENT_THREAD) = 0;
    SetLastError( NO_ERROR );
    return (HANDLE) SnapBuf;
  }

static BOOL GetProcessEntry( char* SnapBuf, LPPROCESSENTRY32 lppe )
// process enumerator
  {
    char*   SnapEntry;
    char    Ustr[8], Astr[8];
    DWORD   pid, Sz;

    if( lppe == NULL || SnapBuf == NULL || SnapBuf == INVALID_HANDLE_VALUE ) {
      SetLastError( ERROR_INVALID_PARAMETER );
      return FALSE;
    }
    if( lppe->dwSize != sizeof( PROCESSENTRY32 ) ) {
      SetLastError( ERROR_INVALID_PARAMETER );
      return FALSE;
    }
    for(;;) { // check current entry and advance to the next
      if( *(DWORD*)(SnapBuf + SNAP_EPROC_CURRENT_PROC) == -1 ) {
        SetLastError( ERROR_NO_MORE_ITEMS );
        return FALSE;
      }
      SnapEntry = SnapBuf + *(DWORD*)(SnapBuf + SNAP_EPROC_CURRENT_PROC);
      pid = *(DWORD*)(SnapEntry + 0x44);
      Sz = *(DWORD*)SnapEntry;
      if( Sz == 0 )
        *(DWORD*)(SnapBuf + SNAP_EPROC_CURRENT_PROC) = -1;
      else
        *(DWORD*)(SnapBuf + SNAP_EPROC_CURRENT_PROC) += Sz;
      if( pid != 0 ) break;
    }
    lppe->cntUsage = 1;
    lppe->th32DefaultHeapID = 0;
    lppe->th32ModuleID = 0;
    lppe->pcPriClassBase = NORMAL_PRIORITY_CLASS;
    lppe->dwFlags = 0;

    lppe->th32ProcessID = pid;
    lppe->cntThreads = *(DWORD*)(SnapEntry + 4);
    lppe->th32ParentProcessID = *(DWORD*)(SnapEntry + 0x48);

    lppe->szExeFile[0] = '\0';
    pRtlInitUnicodeString( Ustr, SnapEntry + 0x88 + 0x40 * lppe->cntThreads );
    pRtlUnicodeStringToAnsiString( Astr, Ustr, TRUE );
    lstrcat( lppe->szExeFile, *(char**)(&Astr[4]) );
    pRtlFreeAnsiString( Astr );

    SetLastError( NO_ERROR );
    return TRUE;
  }

static BOOL WINAPI NtThProcess32First( HANDLE hSnapshot,
                                       LPPROCESSENTRY32 lppe )
  {
    if( hSnapshot != NULL && hSnapshot != INVALID_HANDLE_VALUE ) {
      *(DWORD*)(((char*)hSnapshot) + SNAP_EPROC_CURRENT_PROC)
                                                        = SNAP_RESERVED_SZ;
    }
    return GetProcessEntry( (char*) hSnapshot, lppe );
  }

static BOOL WINAPI NtThProcess32Next( HANDLE hSnapshot, LPPROCESSENTRY32 lppe )
  {
    return GetProcessEntry( (char*) hSnapshot, lppe );
  }

static BOOL GetThreadEntry( char* SnapBuf, LPTHREADENTRY32 lpte )
// thread enumerator
  {
    char*   SnapEntry;
    DWORD   pid, Sz, thr;

    if( lpte == NULL || SnapBuf == NULL || SnapBuf == INVALID_HANDLE_VALUE ) {
      SetLastError( ERROR_INVALID_PARAMETER );
      return FALSE;
    }
    if( lpte->dwSize != sizeof( THREADENTRY32 ) ) {
      SetLastError( ERROR_INVALID_PARAMETER );
      return FALSE;
    }
    for(;;) {  // check current entry and advance to the next
      if( *(DWORD*)(SnapBuf + SNAP_ETHR_CURRENT_PROC) == -1 ) {
        SetLastError( ERROR_NO_MORE_ITEMS );
        return FALSE;
      }
      SnapEntry = SnapBuf + *(DWORD*)(SnapBuf + SNAP_ETHR_CURRENT_PROC);
      pid = *(DWORD*)(SnapEntry + 0x44);
      if( pid != 0 ) {
        // check thread count
        thr = *(DWORD*)(SnapBuf + SNAP_ETHR_CURRENT_THREAD);
        if( thr >= *(DWORD*)(SnapEntry + 4) )
           // no more threads in process - will advance to the next process
          *(DWORD*)(SnapBuf + SNAP_ETHR_CURRENT_THREAD) = 0;
        else {
          *(DWORD*)(SnapBuf + SNAP_ETHR_CURRENT_THREAD) =
                    *(DWORD*)(SnapBuf + SNAP_ETHR_CURRENT_THREAD) + 1;
          break;
        }
      }
      Sz = *(DWORD*)SnapEntry;
      if( Sz == 0 )
        *(DWORD*)(SnapBuf + SNAP_ETHR_CURRENT_PROC) = -1;
      else
        *(DWORD*)(SnapBuf + SNAP_ETHR_CURRENT_PROC) += Sz;
    }
    lpte->cntUsage = 1;
    lpte->tpBasePri = THREAD_PRIORITY_NORMAL;
    lpte->tpDeltaPri = 0;
    lpte->dwFlags = 0;

    lpte->th32ThreadID = *(DWORD*)(SnapEntry + 0xAC + 0x40 * thr);
    lpte->th32OwnerProcessID = pid;

    SetLastError( NO_ERROR );
    return TRUE;
  }

static BOOL WINAPI NtThThread32First( HANDLE hSnapshot, LPTHREADENTRY32 lpte )
  {
    if( hSnapshot != NULL && hSnapshot != INVALID_HANDLE_VALUE ) {
      *(DWORD*)(((char*)hSnapshot) + SNAP_ETHR_CURRENT_PROC)
                                                       = SNAP_RESERVED_SZ;
      *(DWORD*)(((char*)hSnapshot) + SNAP_ETHR_CURRENT_THREAD) = 0;
    }
    return GetThreadEntry( (char*) hSnapshot, lpte );
  }

static BOOL WINAPI NtThThread32Next( HANDLE hSnapshot, LPTHREADENTRY32 lpte )
  {
    return GetThreadEntry( (char*) hSnapshot, lpte );
  }

static BOOL WINAPI NtThCloseHandle( HANDLE hObject )
  {
    if( hObject != NULL && hObject != INVALID_HANDLE_VALUE )
      LocalFree( hObject );
    SetLastError( NO_ERROR );
    return TRUE;
  }

static BOOL WINAPI CallCloseHandle( HANDLE hObject )
  {
    return CloseHandle( hObject );
  }


BOOL IsWindowsNT()
  {
    OSVERSIONINFO  Vi;

    Vi.dwOSVersionInfoSize = sizeof( OSVERSIONINFO );
    if( GetVersionEx( &Vi ) )
      if( Vi.dwPlatformId == VER_PLATFORM_WIN32_NT ) {
        WindowsNT = TRUE;
        return TRUE;
      }
    WindowsNT = FALSE;
    return FALSE;
  }

void ToolhelpInit()
  {
    HMODULE  Th;
    char     ThPath[ MAX_PATH ];

    if( WindowsNT ) {
      pCreateToolhelp32Snapshot = NtThCreateToolhelp32Snapshot;
      pProcess32First           = NtThProcess32First;
      pProcess32Next            = NtThProcess32Next;
      pThread32First            = NtThThread32First;
      pThread32Next             = NtThThread32Next;
      pCloseHandle              = NtThCloseHandle;

      GetSystemDirectory( ThPath, MAX_PATH - lstrlen( NtdllFName ) - 2 );
      if( ThPath[ lstrlen( ThPath ) - 1 ] != Slash[0] )
        lstrcat( ThPath, Slash );
      lstrcat( ThPath, NtdllFName );
      Th = GetModuleHandle( ThPath );
      if( Th == NULL ) return;
      pNtQuerySystemInformation =
          (TNtQuerySystemInformation) GetProcAddress( Th,
                                              "NtQuerySystemInformation" );
      pRtlUnicodeStringToAnsiString =
          (TRtlUnicodeStringToAnsiString) GetProcAddress( Th,
                                              "RtlUnicodeStringToAnsiString" );
      pRtlInitUnicodeString =
          (TRtlInitUnicodeString) GetProcAddress( Th, "RtlInitUnicodeString" );
      pRtlFreeAnsiString =
          (TRtlFreeAnsiString) GetProcAddress( Th, "RtlFreeAnsiString" );
    }
    else {
      GetSystemDirectory( ThPath, MAX_PATH - lstrlen( Kernel32FName ) - 2 );
      if( ThPath[ lstrlen( ThPath ) - 1 ] != Slash[0] ) lstrcat( ThPath, Slash );
      lstrcat( ThPath, Kernel32FName );
      Th = GetModuleHandle( ThPath );
      if( Th == NULL ) return;
      pCreateToolhelp32Snapshot =
          (TCreateToolhelp32Snapshot) GetProcAddress( Th,
                                                "CreateToolhelp32Snapshot" );
      pProcess32First =
          (TProcess32First) GetProcAddress( Th, "Process32First" );
      pProcess32Next =
          (TProcess32Next) GetProcAddress( Th, "Process32Next" );
      pThread32First =
          (TThread32First) GetProcAddress( Th, "Thread32First" );
      pThread32Next =
          (TThread32Next) GetProcAddress( Th, "Thread32Next" );
      pCloseHandle = CallCloseHandle;
    }
  }

HANDLE ThCreateToolhelp32Snapshot( DWORD dwFlags, DWORD th32ProcessID )
  {
    if( pCreateToolhelp32Snapshot == NULL ) {
      SetLastError( ERROR_CALL_NOT_IMPLEMENTED );
      return INVALID_HANDLE_VALUE;
    }
    return pCreateToolhelp32Snapshot( dwFlags, th32ProcessID );
  }

BOOL ThProcess32First( HANDLE hSnapshot, LPPROCESSENTRY32 lppe )
  {
    if( pProcess32First == NULL ) {
      SetLastError( ERROR_CALL_NOT_IMPLEMENTED );
      return FALSE;
    }
    return pProcess32First( hSnapshot, lppe );
  }

BOOL ThProcess32Next( HANDLE hSnapshot, LPPROCESSENTRY32 lppe )
  {
    if( pProcess32Next == NULL ) {
      SetLastError( ERROR_CALL_NOT_IMPLEMENTED );
      return FALSE;
    }
    return pProcess32Next( hSnapshot, lppe );
  }

BOOL ThThread32First( HANDLE hSnapshot, LPTHREADENTRY32 lpte )
  {
    if( pThread32First == NULL ) {
      SetLastError( ERROR_CALL_NOT_IMPLEMENTED );
      return FALSE;
    }
    return pThread32First( hSnapshot, lpte );
  }

BOOL ThThread32Next( HANDLE hSnapshot, LPTHREADENTRY32 lpte )
  {
    if( pThread32Next == NULL ) {
      SetLastError( ERROR_CALL_NOT_IMPLEMENTED );
      return FALSE;
    }
    return pThread32Next( hSnapshot, lpte );
  }

void ThCloseHandle( HANDLE hSnapshot )
  {
    if( pCloseHandle != NULL ) pCloseHandle( hSnapshot );
  }
